
const jwt = require("jsonwebtoken");
require('dotenv').config();
const { COGNITO_CLIENT_ID, AWS_REGION, COGNITO_CLIENT_SECRET } = process.env;
const { CognitoIdentityProviderClient, GetUserCommand, InitiateAuthCommand } = require("@aws-sdk/client-cognito-identity-provider");
const { errorResponse } = require('../utils/utils');

const client = new CognitoIdentityProviderClient({region: AWS_REGION});

const authenticateToken = async(req,res,next) => {
    const auth_Header = req.headers['authorization'];
    const token = auth_Header && auth_Header.split(' ')[1];

    if(!token) return errorResponse(res, 400, "Access Token Required");

    try {
        const decoded_Token = jwt.decode(token);
        // const username = decoded_Token['cognito:username'];
        const currentTime = Math.floor(Date.now() / 1000);

        if (decoded_Token && decoded_Token.exp < currentTime) return errorResponse(res, 401, "Access Token Expired");

        const command = new GetUserCommand({AccessToken: token});

        await client.send(command);

        req.user = decoded_Token;
        next();
    } catch (error) {
        if (error.name === 'NotAuthorizedException' && error.message === 'Access Token has expired') return errorResponse(res, 401, "Access Token Expired");

        if (error.name === 'NotAuthorizedException' && error.message === 'Invalid Access Token') return errorResponse(res, 401, "Invalid Access Token");

        if (error.name === 'NotAuthorizedException' && error.message === 'Access Token has been revoked') return errorResponse(res, 401, "Access Token has been revoked");

        console.log("Server error ",error);
        return errorResponse(res, 500, "Internal server error");
    }
}

const refreshAuthToken = async (refresh_token, res) => {
    try {
        const command = new InitiateAuthCommand({
            AuthFlow: 'REFRESH_TOKEN_AUTH',
            ClientId: COGNITO_CLIENT_ID,
            AuthParameters: {
                REFRESH_TOKEN: refresh_token,
                SECRET_HASH: COGNITO_CLIENT_SECRET
            }
        });

        const result = await client.send(command);

        if (!result || !result.AuthenticationResult) {
            throw new Error("No AuthenticationResult from Cognito");
        }

        return result.AuthenticationResult;

    } catch (error) {
        if (error.name === 'NotAuthorizedException') {
            if (error.message === 'Refresh Token has expired') {
                return errorResponse(res, 401, "Refresh Token Expired");
            }

            if (error.message === 'Invalid Refresh Token') {
                return errorResponse(res, 401, "Invalid Refresh Token");
            }

            if (error.message === 'Refresh Token has been revoked') {
                return errorResponse(res, 401, "Refresh Token has been revoked");
            }
        }

        console.log("refresh errors", error);
        return errorResponse(res, 500, "An error occurred during token refresh");
    }
};

module.exports = { authenticateToken, refreshAuthToken };
