
const express = require('express');
const { createApp, updateApp, getAllApps, getAppById, deleteApp, getTotalApps } = require("../../models/Apps/Apps");
const { errorResponse, successResponse, formatted_date, activityLog, generateRandomNumber, filter_Old_New_Data, saveImage } = require('../../utils/utils');
const Joi = require('joi');
const multer = require('multer');
const dotenv = require('dotenv');

dotenv.config();

const router = express.Router();

const Storage = saveImage();

const upload = multer({ storage: Storage });

const schema = Joi.object().keys({
  name: Joi.string().required(),
  app_url: Joi.string().required(),
  app_description: Joi.string(),
  status: Joi.number().valid(0, 1)
});

async function CreateApp(req, res) {
  try {
    let body = req.body;
    let app_id = generateRandomNumber('AN');

    if (!req.file) return errorResponse(res,400,"No File Uploaded");

    const { error } = schema.validate(body);
    if (error) return errorResponse(res, 400, `Invalid request: ${error.details.map(x => x.message).join(', ')}`);

    const imagePath = `${process.env.DomainName}/assets/${req.file.filename}`;
    const result = await createApp(app_id, imagePath, body.name, body.app_url, body.app_description, body.status, formatted_date);

    if (!result || result instanceof Error) return errorResponse(res, 400, "Failed to create user");

    await activityLog("Added App","App Module", app_id, "POST", null, JSON.stringify(body));

    successResponse(res,"App Created successfully");

  } catch (error) {
    errorResponse(res,500,"Internal server error");
  }
}

async function UpdateApp(req, res) {
  try {
    let app_id = req.query.app_id;
    let body = req.body;

    if (!app_id) return errorResponse(res, 400, "Provide valid App ID");

    const currentApp = await getAppById(app_id);

    if (!currentApp || currentApp.length === 0) return errorResponse(res, 404, "App not found");

    const oldValues = currentApp[0];

    body.picture = req.file ? `${process.env.DomainName}/assets/${req.file.filename}` : oldValues.picture;

    const result = await updateApp(app_id, body.picture, body.name, body.app_url,body.app_description, body.status, formatted_date);
    
    if (!result || result instanceof Error) return errorResponse(res, 400, "Failed to create user");

    if (result.affectedRows === 0) return errorResponse(res, 404, "No changes made");

    // get changed values
    const { changes, new_value_log } = filter_Old_New_Data(oldValues, body);

    await activityLog( "Updated App", "App Module", app_id, "UPDATE", JSON.stringify(changes), JSON.stringify(new_value_log) );

    successResponse(res, "App Updated Successfully");

  } catch (error) {
    errorResponse(res, 500, "Internal server error");
  }
}

async function GetAllApps(req, res) {
  try {
    const Search_Text = req.query.search_text || "";
    const Current_Page = req.query.current_page || 1;
    const Per_Page = req.query.per_page || 100;

    const totalApps = await getTotalApps(Search_Text);

    const result = await getAllApps(Current_Page, Per_Page, Search_Text);

    const total = totalApps[0].total;
    const last_page_number = Math.ceil(total / Per_Page);

    const response_data = {
      current_page: Current_Page,
      per_page: Per_Page,
      total,
      last_page_number,
      result
    };

    successResponse(res,"App details fetched Successfully", response_data);

  } catch (error) {
    console.log(error)
    errorResponse(res,500,"Internal server error");
  }
}

async function GetAppById(req, res) {
  try {
    const app_id = req.query.app_id;

    if(!app_id){
      return errorResponse(res,404,"Record Not found");
    }

    const result = await getAppById(app_id);

    if (result.length > 0){
      successResponse(res,"App detail fetched Successfully", result[0]);
    } else{
      errorResponse(res,404,"Record not found");
    }

  } catch (error) {
    errorResponse(res,500,"Internal server error");
  }
}

async function DeleteApp(req, res) {
  try {
    let app_id = req.query.app_id;

    if(!app_id) return errorResponse(res,404,"ID not found");

    const result = await deleteApp(app_id);

    if (result.affectedRows === 0) {
      return errorResponse(res,404,"App not found");
    } else {
      await activityLog("Deleted App","App Module", app_id, "DELETE", null, null);

      successResponse(res,"App deleted successfully");
    }

  } catch (error) {
    errorResponse(res,500,"Internal server error");
  }
}

router.post("/add-app", upload.single('picture'), CreateApp);
router.put("/update-app", upload.single('picture'), UpdateApp);
router.get("/get-app-list", GetAllApps);
router.get("/get-single-app", GetAppById);
router.delete("/delete-app", DeleteApp);

module.exports = router;
