
const express = require('express');
const multer = require('multer');
const { createUser, updateUser, getAllUsers, getUserById, deleteUser, getUserInfo, getTotalUsers } = require("../../models/Users/Users");
const { errorResponse, successResponse, activityLog, formatted_date, generateRandomNumber, filter_Old_New_Data, saveImage, emailRegex, mobileRegex } = require('../../utils/utils');
const Joi = require("joi");
const country_data = require("../../Constants/countries.json");
const state_data = require("../../Constants/states.json");
const city_data = require("../../Constants/cities.json");

const router = express.Router();

// function for storing image in folder
const Storage = saveImage();

const upload = multer({ storage: Storage });

const schema = Joi.object().keys({
  first_name: Joi.string().required(),
  last_name: Joi.string().required(),
  country: Joi.number().required(),
  state: Joi.number().required(),
  city: Joi.number().required(),
  street_1: Joi.string().required(),
  street_2: Joi.string().empty(''),
  zip_code: Joi.string().required(),
  email: Joi.string().required(),
  dial_code: Joi.string().required(),
  mobile: Joi.string().required(),
  user_role: Joi.number().valid(0, 1),
  status: Joi.number().valid(0, 1)
});

async function CreateUser(req, res) {
  try {
    let body = req.body;

    const { error } = schema.validate(body);
    if (error) return errorResponse(res, 400, `Invalid request: ${error.details.map(x => x.message).join(', ')}`);

    let randomNumber = generateRandomNumber('UN');

    if (!req.file) return errorResponse(res, 400, "No File Uploaded");

    const imagePath = `${process.env.DomainName}/assets/${req.file.filename}`;

    const result = await createUser(randomNumber, body.first_name, body.last_name, imagePath, body.country, body.state, body.city, body.street_1, body.street_2, body.zip_code, body.email, body.dial_code, body.mobile, body.user_role, body.status, formatted_date);

    if (!result || result instanceof Error) {
      if (result.message.includes("Duplicate entry")) {
        return errorResponse(res, 400, "Email already exists, please use a different email.");
      }
      return errorResponse(res, 400, "Failed to create user");
    }

    await activityLog("Added User", "User Module", randomNumber, "POST", null, JSON.stringify(body));

    successResponse(res, "User Created successfully");

  } catch (error) {
    return errorResponse(res, 500, error.message);
  }
}

async function UpdateUser(req, res) {
  try {
    let user_id = req.query.user_id;
    let body = req.body;

    if (!user_id) return errorResponse(res, 400, "Provide valid USER ID");

    const currentUser = await getUserById(user_id);

    if (!currentUser || currentUser.length === 0) return errorResponse(res, 404, "User not found");

    const oldValues = currentUser[0];

    body.picture = req.file ? `${process.env.DomainName}/assets/${req.file.filename}` : oldValues.picture;

    const result = await updateUser(body.first_name, body.last_name, body.picture, body.country, body.state, body.city, body.street_1, body.street_2, body.zip_code, body.email, body.dial_code, body.mobile, body.user_role, body.status, formatted_date, user_id);

    if (result.affectedRows === 0) return errorResponse(res, 404, "No changes made");

    if (!result || result instanceof Error) {
      if (result.message.includes("Duplicate entry")) {
        return errorResponse(res, 400, "Email already exists, please use a different email.");
      }
      return errorResponse(res, 400, "Failed to create user");
    }

    // get changed values
    const { changes, new_value_log } = filter_Old_New_Data(oldValues, body);

    await activityLog("Updated User", "User Module", user_id, "UPDATE", JSON.stringify(changes), JSON.stringify(new_value_log));

    successResponse(res, "User Updated Successfully");

  } catch (error) {
    return errorResponse(res, 500, error.message);
  }
}

async function GetAllUsers(req, res) {
  try {
    const Search_Text = req.query.search_text || "";
    const Current_Page = req.query.current_page || 1;
    const Per_Page = req.query.per_page || 10;

    const totalUsers = await getTotalUsers(Search_Text);

    const users = await getAllUsers(Current_Page, Per_Page, Search_Text);

    const total = totalUsers[0].total;
    const last_page_number = Math.ceil(total / Per_Page);

    const response_data = {
      current_page: Current_Page,
      per_page: Per_Page,
      total,
      last_page_number,
      users
    };

    successResponse(res, "User Details Fetched Successfully", response_data);

  } catch (error) {
    console.log(error)
    return errorResponse(res, 500, "Internal server error");
  }
}

async function GetUserById(req, res) {
  try {
    let user_id = req.query.user_id;

    if (!user_id) {
      return errorResponse(res, 404, "Record Not found");
    }

    const result = await getUserInfo(user_id);
    let { name:countryName } = country_data.find((item) => item.id == result[0].country);
    let { name:stateName } = state_data.find((item) => item.id == result[0].state);
    let { name:cityName } = city_data.find((item) => item.id == result[0].city);

    if (result.length > 0) {
      successResponse(res, "User Detail Fetched Successfully", {...result[0],countryName,stateName,cityName});
    } else {
      errorResponse(res, 500, "Record not found");
    }

  } catch (error) {
    return errorResponse(res, 500, "Internal server error");
  }
}

async function DeleteUser(req, res) {
  try {
    let user_id = req.query.user_id;

    if (!user_id) return errorResponse(res, 404, "ID not found");

    const result = await deleteUser(user_id);

    if (result.affectedRows === 0) {
      return errorResponse(res, 404, "User not found");
    } else {
      await activityLog("Deleted User", "User Module", user_id, "DELETE", null, null);

      successResponse(res, "User deleted successfully");
    }

  } catch (error) {
    return errorResponse(res, 500, error.message);
  }
}

async function updateAssignedApps (req,res) {
  try {
    let user_id = req.query.user_id;
    let body = req.body;

    if (!user_id) return errorResponse(res, 400, "Provide valid USER ID");

    const result = await UpdateAssignedApp(JSON.stringify(body.assigned_apps) ,user_id);

    if (result.affectedRows === 0) return errorResponse(res, 404, "No changes made");

    if (!result || result instanceof Error) {
      if (result.message.includes("Duplicate entry")) {
        return errorResponse(res, 400, "Email already exists, please use a different email.");
      }
      return errorResponse(res, 400, "Failed to create user");
    }

    successResponse(res, "User Updated Successfully");

  } catch (error) {
    console.log("Assigned apps error ", error);
    errorResponse(res, 500, "Internal server error");
  }
}

router.post("/add-user", upload.single('picture'), CreateUser);
router.put("/update-user", upload.single('picture'), UpdateUser);
router.get("/get-all-users", GetAllUsers);
router.get("/get-single-user", GetUserById);
router.delete("/delete-user", DeleteUser);

router.put("/update-assigned-apps", upload.none(),updateAssignedApps);

module.exports = router;
